from enum import Enum
from pydantic import BaseModel, Field, EmailStr, validator
from bson import ObjectId
from datetime import datetime
from typing import Optional, List

class Status(str, Enum):
    ACTIVE = "active"
    INACTIVE = "inactive"
    MAINTENANCE = "maintenance"

class Address(BaseModel):
    street: str = Field(..., min_length=2, max_length=100)
    city: str = Field(..., min_length=2, max_length=50)
    state: str = Field(..., min_length=2, max_length=50)
    postal_code: str = Field(..., min_length=3, max_length=20)
    country: str = Field(..., min_length=2, max_length=50)

class VendorType(str, Enum):
    VENDOR = "vendor"
    DEALER = "dealer"
    MANUFACTURER = "manufacturer"
    SERVICE_CENTER = "service_center"

class VendorBase(BaseModel):
    vendor_name: str = Field(..., min_length=2, max_length=100)
    address: Address
    phone: str = Field(..., min_length=5, max_length=20)
    contact_person_name: str = Field(..., min_length=2, max_length=50)
    email: EmailStr
    vendor_type: VendorType
    # logo: Optional[str] = Field(None)
    license_number: str = Field(..., min_length=2, max_length=50)
    # tax_id: Optional[str] = Field(None, min_length=2, max_length=50)
    # website: Optional[str] = Field(None)
    tax_id: Optional[str] = Field(None, min_length=0, max_length=50)  # Make optional
    website: Optional[str] = Field(None, min_length=0)  # Make optional
    logo: Optional[str] = Field(None, min_length=0)  # Make optional
    status: Status = Field(default=Status.ACTIVE)
       

class VendorCreate(VendorBase):
    account_id: str = Field("", min_length=0)
    user_id: str = Field("", min_length=0)
    vendor_id: str
    v_id: int

class Vendor(VendorCreate):
    id: str = Field(default=None, alias="_id")
    created_date: datetime = Field(default_factory=datetime.utcnow)
    updated_date: datetime = Field(default_factory=datetime.utcnow)

    class Config:
        arbitrary_types_allowed = True
        json_encoders = {ObjectId: str}
        allow_population_by_field_name = True

    @validator('id', pre=True)
    def convert_objectid(cls, v):
        if isinstance(v, ObjectId):
            return str(v)
        return v

class VendorResponse(BaseModel):
    account_id: Optional[str] = Field(None, min_length=0)
    user_id: Optional[str] = Field(None, min_length=0)
    vendor_name: str
    address: Address
    phone: str
    contact_person_name: str
    email: str
    vendor_type: str
    logo: Optional[str]
    license_number: str
    tax_id: Optional[str]
    website: Optional[str]
    status: Optional[str]
    created_date: datetime
    updated_date: datetime
    vendor_id: Optional[str] = None
    v_id: Optional[int] = None

class VendorResponseList(BaseModel):
    total_count: int
    vendors: List[VendorResponse]

class VendorUpdate(BaseModel):
    vendor_name: Optional[str] = Field(None, min_length=2, max_length=100)
    address: Optional[Address] = None
    phone: Optional[str] = Field(None, min_length=5, max_length=20)
    contact_person_name: Optional[str] = Field(None, min_length=2, max_length=50)
    email: Optional[EmailStr] = None
    vendor_type: Optional[VendorType] = None
    logo: Optional[str] = None
    license_number: Optional[str] = Field(None, min_length=2, max_length=50)
    tax_id: Optional[str] = Field(None, min_length=2, max_length=50)
    website: Optional[str] = None
    status: Status = Field(default=Status.ACTIVE)